/**
* \file: lock_cmd.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: authorization level daemon
*
* \author: Rexaline Xavier  /  RexalineInfancia.Xavier@in.bosch.com
*
* \copyright (c) 2017 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>

#include "common.h"
#include "encryption/signature.h"

#include "util/helper.h"
#include "util/logger.h"
#include "util/cmdline_parser.h"
#include "util/cfgfile_parser.h"

static cfg_item_t help_cfg = {.spec = &help_cfg_spec, .cmdline = &help_cmdline_spec };

static error_code_t lock_cmd_init(void);
static error_code_t lock_cmd_parse_args(const char *binary_name, int argc, char *argv[]);
static error_code_t lock_cmd_start(void);
static void lock_cmd_deinit(void);

static cfg_item_t *const lock_cmd_cfgs[] = {
		&help_cfg
};

command_vtable_t lock_cmd_vtable = {
        .command = "lock",
        .command_description = "Locks the device. ALD will go to closed state",
        .init = lock_cmd_init,
        .parse_args = lock_cmd_parse_args,
        .start = lock_cmd_start,
        .deinit = lock_cmd_deinit
};

static void lock_cmd_print_usage(const char *binary_name)
{
	printf("\nUsage: %s %s \n\n",binary_name, lock_cmd_vtable.command);
}

static void lock_cmd_print_help(const char *binary_name)
{
	printf("\nAuthorization Level Daemon Change Level Utility\n"
			"\t- Used to change the system security level to 0 (locked) with the help of the ADIT authorization level daemon\n\n");
	printf("Usage: %s %s \n\n",binary_name, lock_cmd_vtable.command);
	printf("This command allows to set the device into the locked state. On receiving\n"
		   "this command ALD does a level change to to system security level 0. Thus closing the device.\n\n");
}

static error_code_t lock_cmd_init(void)
{
    logger_init_console_only(LOGGER_LEVEL_ERROR);

    return common_init();
}

static error_code_t lock_cmd_parse_args(const char *binary_name, int argc, char *argv[])
{
	error_code_t result = RESULT_OK;
	int lock_cmd_cfgs_cnt = ARRAY_SIZE(lock_cmd_cfgs);

	helper_items_init(lock_cmd_cfgs, lock_cmd_cfgs_cnt);

	result = cmdline_parser_parse_cmd_line(lock_cmd_cfgs, lock_cmd_cfgs_cnt, argv, argc);

	if (result == RESULT_INVALID_ARGS)
	{
		lock_cmd_print_usage(binary_name);
		return RESULT_INVALID_ARGS;
	}

	if (helper_get_flag(&help_cfg))
	{
		lock_cmd_print_help(binary_name);
		return RESULT_HELP_PRINTED;
	}
	return result;
}

static error_code_t lock_cmd_start(void)
{
	Change_level *change_level_iface;
	error_code_t result;

	result = common_connect_ALD(&change_level_iface);
	if (result == RESULT_OK)
	{
		printf("--------------------- Locking the device -----------------------------------------------------\n");
		result = common_lock_device(change_level_iface);
	}
	common_disconnect_ald(&change_level_iface);

	if (result == RESULT_OK)
		printf("Device locked successfully.\n");
	else
		printf("Locking the device failed.\n");

	return result;
}

static void lock_cmd_deinit(void)
{
    common_deinit();
}

